// LinkedIn Lead Extractor Content Script - Server Mode
// COMPLETE implementation - ALL functions from content.js adapted for server mode

console.log('🚀 LinkedIn Lead Extractor Content Script - Server Mode');

// Prevent multiple script injections
if (window.linkedinExtractorLoaded) {
  console.log('⚠️ LinkedIn Extractor already loaded, skipping duplicate injection');
} else {
  window.linkedinExtractorLoaded = true;
  
  let isExtracting = false;
  let extractionId = null;

  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    console.log('📨 Content script message:', request.action);
    
    if (request.action === 'ping') {
      sendResponse({ pong: true });
      return false;
    }
    
    if (request.action === 'detectCountry') {
      const country = detectCountryFromPage();
      sendResponse({ country: country });
      return false;
    }
    
    if (request.action === 'extractLeads') {
      if (isExtracting) {
        console.log('⚠️ Extraction already in progress');
        sendResponse({ 
          success: false, 
          error: 'Extraction already in progress',
          leadsCount: 0,
          savedToDatabase: 0
        });
        return false;
      }
      
      const maxPages = request.maxPages || 1;
      const workerID = request.workerID || 'Unknown';
      const country = request.country || 'auto-detect';
      
      console.log(`🚀 Starting extraction: ${maxPages} pages for worker ${workerID}`);
      
      isExtracting = true;
      extractionId = Date.now();
      const currentExtractionId = extractionId;
      
      let responseSent = false;
      
      const safeSendResponse = (data) => {
        if (!responseSent) {
          try {
            if (isExtensionContextValid()) {
              sendResponse(data);
              responseSent = true;
            }
          } catch (error) {
            console.error('❌ Error sending response:', error);
          }
        }
      };
      
      (async () => {
        try {
          const result = await extractLeadsToServer(maxPages, workerID, country);
          
          if (extractionId === currentExtractionId && !responseSent) {
            safeSendResponse({ 
              success: true, 
              leadsCount: result.totalLeads,
              pagesProcessed: result.pagesProcessed,
              savedToDatabase: result.savedCount,
              duplicatesFound: result.duplicatesCount
            });
          }
        } catch (error) {
          console.error('❌ Extraction error:', error);
          if (extractionId === currentExtractionId && !responseSent) {
            safeSendResponse({ 
              success: false, 
              error: error.message || 'Extraction failed',
              leadsCount: 0,
              savedToDatabase: 0
            });
          }
        } finally {
          if (extractionId === currentExtractionId) {
            isExtracting = false;
            extractionId = null;
          }
        }
      })();
      
      return true;
    }
    
    sendResponse({ success: false, error: 'Unknown action' });
    return false;
  });
}

// Save leads to localStorage for recovery - ASYNC, NON-BLOCKING
function saveLeadsToLocalStorage(leads, workerID, country, currentPage, pagesProcessed) {
  const saveOperation = () => {
    try {
      const backupKey = 'linkedin_leads_current_extraction';
      const backupData = {
        leads: leads,
        workerID: workerID,
        country: country,
        currentPage: currentPage,
        pagesProcessed: pagesProcessed,
        timestamp: new Date().toISOString(),
        count: leads.length
      };
      try {
        const jsonData = JSON.stringify(backupData);
        localStorage.setItem(backupKey, jsonData);
        if (pagesProcessed % 10 === 0) {
          console.log(`💾 [BACKGROUND] Saved ${leads.length} leads to localStorage (page ${currentPage})`);
        }
      } catch (jsonError) {
        if (pagesProcessed % 20 === 0) {
          console.error('❌ JSON stringify error:', jsonError);
        }
      }
    } catch (e) {
      if (pagesProcessed % 20 === 0) {
        console.error('❌ Could not save to localStorage:', e);
      }
    }
  };
  
  if (window.requestIdleCallback) {
    requestIdleCallback(saveOperation, { timeout: 1000 });
  } else {
    setTimeout(saveOperation, 0);
  }
  
  return true;
}

function loadSavedLeadsFromLocalStorage() {
  try {
    const backupKey = 'linkedin_leads_current_extraction';
    const saved = localStorage.getItem(backupKey);
    if (saved) {
      const data = JSON.parse(saved);
      const backupTime = new Date(data.timestamp);
      const now = new Date();
      const hoursDiff = (now - backupTime) / (1000 * 60 * 60);
      if (hoursDiff < 24) {
        return data;
      } else {
        localStorage.removeItem(backupKey);
        return null;
      }
    }
    return null;
  } catch (e) {
    console.error('❌ Could not load from localStorage:', e);
    return null;
  }
}

function clearSavedLeadsFromLocalStorage() {
  try {
    localStorage.removeItem('linkedin_leads_current_extraction');
    return true;
  } catch (e) {
    return false;
  }
}

// Check extension context validity
function isExtensionContextValid() {
  try {
    return typeof chrome !== 'undefined' && 
           chrome.runtime && 
           chrome.runtime.id && 
           typeof chrome.runtime.sendMessage === 'function';
  } catch (error) {
    return false;
  }
}

// Detect country from page
function detectCountryFromPage() {
  try {
    console.log('🔍 Looking for Company headquarters location filter...');
    
    const headquartersSection = document.querySelector('[data-x-search-filter="COMPANY_HEADQUARTERS"]');
    
    if (!headquartersSection) {
      const locationPills = document.querySelectorAll('[aria-label*="Current filters"] button, .search-reusables__filter-pill-button');
      for (const pill of locationPills) {
        const text = pill.textContent.trim();
        if (text.includes('United States')) return 'United States';
        if (text.includes('United Kingdom')) return 'United Kingdom';
        if (text.includes('Canada')) return 'Canada';
        if (text.includes('Australia')) return 'Australia';
        if (text.includes('Germany')) return 'Germany';
        if (text.includes('France')) return 'France';
        if (text.includes('Netherlands')) return 'Netherlands';
        if (text.includes('India')) return 'India';
        if (text.includes('Spain')) return 'Spain';
        if (text.includes('Italy')) return 'Italy';
        if (text.includes('Brazil')) return 'Brazil';
        if (text.includes('Mexico')) return 'Mexico';
      }
      return '';
    }
    
    console.log('✅ Found Company headquarters location filter section');
    
    const selectedPills = headquartersSection.querySelectorAll('.artdeco-pill--selected .nowrap-ellipsis, .artdeco-pill--green .nowrap-ellipsis');
    
    for (const pill of selectedPills) {
      const text = pill.textContent?.trim();
      if (text && text.length > 2) {
        console.log('🌍 Found country in headquarters filter:', text);
        return text;
      }
    }
    
    console.log('❌ No selected country found in headquarters filter');
    return '';
    
  } catch (error) {
    console.error('Error detecting country:', error);
    return '';
  }
}

// ULTRA-AGGRESSIVE scroll to top
async function ensureAtTop() {
  console.log('📍 ULTRA: Forcing scroll to absolute top...');
  
  for (let attempt = 0; attempt < 20; attempt++) {
    window.scrollTo(0, 0);
    window.scrollTo({ top: 0, left: 0, behavior: 'instant' });
    document.documentElement.scrollTop = 0;
    document.body.scrollTop = 0;
    window.scroll(0, 0);
    
    const mainContainers = document.querySelectorAll('main, .application-outlet, .scaffold-layout__content');
    mainContainers.forEach(container => {
      if (container.scrollTop > 0) {
        container.scrollTop = 0;
      }
    });
    
    const allElements = document.querySelectorAll('*');
    allElements.forEach(el => {
      if (el.scrollTop > 0) {
        el.scrollTop = 0;
      }
    });
    
    await new Promise(resolve => setTimeout(resolve, 50 + (attempt * 10)));
    
    const currentScrollTop = window.pageYOffset || document.documentElement.scrollTop || document.body.scrollTop;
    console.log(`📍 ULTRA attempt ${attempt + 1}: Position ${currentScrollTop}px`);
    
    if (currentScrollTop <= 1) {
      console.log('✅ ULTRA: Perfect - at absolute top!');
      
      await new Promise(resolve => setTimeout(resolve, 200));
      const check1 = window.pageYOffset || document.documentElement.scrollTop;
      
      await new Promise(resolve => setTimeout(resolve, 200));
      const check2 = window.pageYOffset || document.documentElement.scrollTop;
      
      if (check1 <= 1 && check2 <= 1) {
        console.log('✅ ULTRA: Triple verified - staying at top!');
        return true;
      } else {
        console.log(`⚠️ ULTRA: Scroll drifted (${check1}px, ${check2}px), retrying...`);
      }
    }
  }
  
  // Final nuclear approach
  console.log('🔥 ULTRA: FINAL NUCLEAR SCROLL');
  for (let i = 0; i < 10; i++) {
    window.scrollTo(0, 0);
    document.documentElement.scrollTop = 0;
    document.body.scrollTop = 0;
    await new Promise(resolve => setTimeout(resolve, 50));
  }
  
  return true;
}

// FAST extraction without duplicate checking
async function extractPageFast(currentPage) {
  const allExtractedData = [];
  
  console.log(`⚡ FAST: Extracting page ${currentPage} - NO DUPLICATE CHECKS`);
  
  await ensureAtTop();
  await new Promise(resolve => setTimeout(resolve, 300));
  
  let lastProfileCount = 0;
  let stableCount = 0;
  
  while (stableCount < 3) {
    const profileContainers = document.querySelectorAll('li.artdeco-list__item');
    
    for (let i = lastProfileCount; i < profileContainers.length; i++) {
      const container = profileContainers[i];
      
      if (container.getAttribute('data-processed')) continue;
      
      if (i < 2) {
        console.log(`📋 FAST: Processing top profile ${i + 1} (always loaded)`);
        
        const leadElement = container.querySelector('[data-x-search-result="LEAD"]');
        
        if (leadElement) {
          const lead = extractLead(leadElement);
          
          if (lead && lead.fullName && lead.company) {
            allExtractedData.push(lead);
            markProfile(container, 'new');
            console.log(`📋 FAST Extracted (top): ${lead.fullName} - ${lead.company}`);
          } else {
            markProfile(container, 'failed');
          }
        } else {
          markProfile(container, 'failed');
        }
      } else {
        container.scrollIntoView({ behavior: 'smooth', block: 'center' });
        await new Promise(resolve => setTimeout(resolve, 200));
        
        const leadElement = container.querySelector('[data-x-search-result="LEAD"]');
        
        if (leadElement) {
          const lead = extractLead(leadElement);
          
          if (lead && lead.fullName && lead.company) {
            allExtractedData.push(lead);
            markProfile(container, 'new');
            console.log(`📋 FAST Extracted: ${lead.fullName} - ${lead.company}`);
          } else {
            markProfile(container, 'failed');
          }
        } else {
          markProfile(container, 'failed');
        }
      }
      
      container.setAttribute('data-processed', 'true');
    }
    
    lastProfileCount = profileContainers.length;
    
    window.scrollBy(0, window.innerHeight * 0.6);
    await new Promise(resolve => setTimeout(resolve, 900));
    
    const newProfileContainers = document.querySelectorAll('li.artdeco-list__item');
    
    if (newProfileContainers.length > profileContainers.length) {
      stableCount = 0;
    } else {
      stableCount++;
    }
    
    const paginationVisible = document.querySelector('.artdeco-pagination')?.getBoundingClientRect();
    if (paginationVisible && paginationVisible.top < window.innerHeight) {
      break;
    }
  }
  
  console.log(`📋 FAST: Extracted ${allExtractedData.length} leads (no duplicate checking)`);
  
  return {
    newLeads: allExtractedData,
    duplicates: 0,
    failed: 0
  };
}

function extractLead(element) {
  try {
    // Extract name
    let fullName = '';
    const nameSelectors = [
      'span[data-anonymize="person-name"]',
      '.artdeco-entity-lockup__title a span',
      '.entity-result__title-text a span[aria-hidden="true"]',
      '.entity-result__title-text span',
      'a[data-control-name="search_srp_result"] span[aria-hidden="true"]'
    ];
    
    for (const selector of nameSelectors) {
      const nameEl = element.querySelector(selector);
      if (nameEl && nameEl.textContent.trim()) {
        fullName = nameEl.textContent.trim();
        break;
      }
    }
    
    if (!fullName) return null;
    
    fullName = fullName.replace(/[^\w\s\-'\.]/g, ' ').replace(/\s+/g, ' ').trim();
    if (fullName.length < 2) return null;
    
    // Extract company
    let company = '';
    const companySelectors = [
      'a[data-anonymize="company-name"]',
      '.artdeco-entity-lockup__subtitle a',
      '.entity-result__primary-subtitle',
      '.entity-result__secondary-subtitle'
    ];
    
    for (const selector of companySelectors) {
      const companyEl = element.querySelector(selector);
      if (companyEl && companyEl.textContent.trim()) {
        company = companyEl.textContent.trim()
          .replace(/^at\s+/i, '')
          .replace(/\s+•.*$/, '')
          .trim();
        if (company && company.length > 1) break;
      }
    }
    
    if (!company) return null;
    
    company = company.replace(/[^\w\s\-&'\.]/g, ' ').replace(/\s+/g, ' ').trim();
    if (company.length < 2) return null;
    
    // Extract location/state
    let state = '';
    const locationSelectors = [
      '[data-anonymize="location"]',
      '.entity-result__secondary-subtitle',
      '.artdeco-entity-lockup__subtitle .visually-hidden',
      '.entity-result__summary .visually-hidden',
      '.t-12[data-anonymize="location"]',
      'span[data-anonymize="location"]'
    ];
    
    for (const selector of locationSelectors) {
      const locationEl = element.querySelector(selector);
      if (locationEl && locationEl.textContent.trim()) {
        const locationText = locationEl.textContent.trim();
        if (locationText.length > 3 && 
            (locationText.includes(',') || locationText.includes('United States') || 
             locationText.includes('Metroplex') || locationText.includes('Area') ||
             locationText.includes('Greater') || locationText.includes('Metro') ||
             locationText.match(/^[A-Za-z\s\-,\.]+$/))) {
          state = locationText;
          console.log(`📍 Found location with selector "${selector}": ${state}`);
          break;
        }
      }
    }
    
    if (state) {
      state = state.replace(/[^\w\s\-,\.]/g, ' ').replace(/\s+/g, ' ').trim();
    }
    
    if (state) {
      console.log(`📍 LOCATION EXTRACTED: "${state}" for ${fullName} at ${company}`);
    } else {
      console.log(`⚠️ NO LOCATION found for ${fullName} at ${company}`);
    }
    
    // Parse country from location
    let country = 'Unknown';
    let parsedState = '';
    
    if (state) {
      const locationParts = state.split(',').map(p => p.trim());
      
      if (locationParts.length === 1) {
        country = locationParts[0];
        parsedState = '';
      } else if (locationParts.length > 1) {
        country = locationParts[locationParts.length - 1];
        parsedState = locationParts.slice(0, -1).join(', ');
      }
    }
    
    const leadId = `LEAD_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    
    return {
      leadId: leadId,
      fullName: fullName,
      company: company,
      state: parsedState || '',
      country: country,
      extractedAt: new Date().toISOString()
    };
    
  } catch (error) {
    console.error('Error extracting lead:', error);
    return null;
  }
}

function markProfile(container, status) {
  container.setAttribute('data-processed', 'true');
  
  const existingMarker = container.querySelector('.extractor-marker');
  if (existingMarker) {
    existingMarker.remove();
  }
  
  const marker = document.createElement('div');
  marker.className = 'extractor-marker';
  
  let backgroundColor, emoji, title;
  
  if (status === 'new') {
    backgroundColor = '#28a745';
    emoji = '✓';
    title = 'NEW lead - saved to server';
  } else if (status === 'duplicate') {
    backgroundColor = '#ffc107';
    emoji = '🔄';
    title = 'DUPLICATE - already exists';
  } else {
    backgroundColor = '#dc3545';
    emoji = '✗';
    title = 'FAILED - invalid data';
  }
  
  marker.style.cssText = `
    position: absolute;
    top: 5px;
    right: 5px;
    width: 20px;
    height: 20px;
    border-radius: 50%;
    background: ${backgroundColor};
    color: white;
    font-size: 12px;
    font-weight: bold;
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    font-family: Arial, sans-serif;
  `;
  marker.textContent = emoji;
  marker.title = title;
  
  const containerStyle = window.getComputedStyle(container);
  if (containerStyle.position === 'static') {
    container.style.position = 'relative';
  }
  
  container.appendChild(marker);
}

function addStatsIndicator(maxPages) {
  removeStats();
  
  const indicator = document.createElement('div');
  indicator.id = 'linkedin-extractor-persistent-stats';
  indicator.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: linear-gradient(135deg, #0073b1, #005885);
    color: white;
    padding: 20px;
    border-radius: 12px;
    font-size: 14px;
    z-index: 10000;
    font-family: Arial, sans-serif;
    box-shadow: 0 8px 32px rgba(0,0,0,0.3);
    min-width: 400px;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255,255,255,0.1);
  `;
  
  indicator.innerHTML = `
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 12px;">
      <div style="font-weight: bold; font-size: 16px;">🚀 LinkedIn → Server</div>
      <div id="minimize-stats" style="cursor: pointer; font-size: 14px; opacity: 0.7; padding: 4px; background: rgba(255,255,255,0.1); border-radius: 4px;" title="Minimize">−</div>
    </div>
    
    <div id="stats-content">
      <div style="margin-bottom: 12px;">
        <div id="progress-text" style="font-weight: 600; margin-bottom: 4px;">Preparing...</div>
        <div style="background: rgba(255,255,255,0.3); height: 8px; border-radius: 4px;">
          <div id="progress-bar" style="background: #90EE90; height: 100%; border-radius: 4px; width: 0%; transition: width 0.5s; box-shadow: 0 0 10px rgba(144,238,144,0.5);"></div>
        </div>
        <div id="progress-details" style="font-size: 11px; margin-top: 4px; opacity: 0.9;">Starting extraction...</div>
      </div>
      
      <div style="border-top: 1px solid rgba(255,255,255,0.2); padding-top: 12px;">
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px; font-size: 13px; margin-bottom: 12px;">
          <div>
            <div style="color: #90EE90; font-weight: bold; margin-bottom: 4px;">
              <span style="font-size: 16px;">✓</span> <span id="new-count">NEW: 0</span>
            </div>
            <div style="color: #FFD700; font-weight: bold; margin-bottom: 4px;">
              <span style="font-size: 16px;">🔄</span> <span id="duplicate-count">DUPLICATES: 0</span>
            </div>
          </div>
          <div>
            <div style="color: #FF6B6B; font-weight: bold; margin-bottom: 4px;">
              <span style="font-size: 16px;">✗</span> <span id="failed-count">FAILED: 0</span>
            </div>
            <div style="color: #87CEEB; font-weight: bold; margin-bottom: 4px;">
              <span style="font-size: 16px;">💾</span> <span id="saved-count">SAVED: 0</span>
            </div>
          </div>
        </div>
        
        <div style="border-top: 1px solid rgba(255,255,255,0.1); padding-top: 12px;">
          <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 8px; font-size: 12px;">
            <div>⏱️ <span id="extraction-time">00:00</span></div>
            <div>📄 Page: <span id="current-page-display">1</span>/<span id="total-pages-display">${maxPages}</span></div>
          </div>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(indicator);
  
  const minimizeBtn = document.getElementById('minimize-stats');
  if (minimizeBtn) {
    minimizeBtn.addEventListener('click', () => {
      const content = document.getElementById('stats-content');
      
      if (content.style.display === 'none') {
        content.style.display = 'block';
        minimizeBtn.textContent = '−';
        minimizeBtn.title = 'Minimize';
      } else {
        content.style.display = 'none';
        minimizeBtn.textContent = '+';
        minimizeBtn.title = 'Expand';
      }
    });
  }
  
  const startTime = Date.now();
  const timer = setInterval(() => {
    const elapsed = Math.floor((Date.now() - startTime) / 1000);
    const minutes = Math.floor(elapsed / 60);
    const seconds = elapsed % 60;
    const timeDisplay = document.getElementById('extraction-time');
    if (timeDisplay) {
      timeDisplay.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    }
  }, 1000);
  
  indicator.setAttribute('data-timer-id', timer);
}

function updateStats(current, max, message, savedCount, stats = {}) {
  const progressText = document.getElementById('progress-text');
  const progressBar = document.getElementById('progress-bar');
  const progressDetails = document.getElementById('progress-details');
  const currentPageDisplay = document.getElementById('current-page-display');
  const newCount = document.getElementById('new-count');
  const duplicateCount = document.getElementById('duplicate-count');
  const failedCount = document.getElementById('failed-count');
  const savedCountDisplay = document.getElementById('saved-count');
  
  if (progressText && progressBar && progressDetails) {
    progressText.textContent = `Page ${current} of ${max}`;
    progressBar.style.width = `${(current / max) * 100}%`;
    progressDetails.textContent = message || 'Processing...';
  }
  
  if (currentPageDisplay) {
    currentPageDisplay.textContent = current;
  }
  
  if (newCount) newCount.textContent = `NEW: ${stats.new || 0}`;
  if (duplicateCount) duplicateCount.textContent = `DUPLICATES: ${stats.duplicates || 0}`;
  if (failedCount) failedCount.textContent = `FAILED: ${stats.failed || 0}`;
  if (savedCountDisplay) savedCountDisplay.textContent = `SAVED: ${savedCount || 0}`;
}

function removeStats() {
  const stats = document.getElementById('linkedin-extractor-persistent-stats');
  if (stats) {
    const timerId = stats.getAttribute('data-timer-id');
    if (timerId) {
      clearInterval(parseInt(timerId));
    }
    stats.remove();
  }
  
  const markers = document.querySelectorAll('.extractor-marker');
  markers.forEach(marker => marker.remove());
  
  const processedContainers = document.querySelectorAll('[data-processed]');
  processedContainers.forEach(container => {
    container.removeAttribute('data-processed');
    container.style.position = '';
  });
}

async function checkPageDuplicates(leads) {
  try {
    if (!leads || leads.length === 0) {
      console.log('⚠️ No leads to check for duplicates');
      return [];
    }
    
    if (!isExtensionContextValid()) {
      if (!checkPageDuplicates._contextWarningLogged) {
        console.warn('⚠️ Extension context invalid - assuming all leads are NEW');
        checkPageDuplicates._contextWarningLogged = true;
        setTimeout(() => { checkPageDuplicates._contextWarningLogged = false; }, 5000);
      }
      return leads.map(() => false);
    }
    
    console.log(`🔍 Checking ${leads.length} leads for duplicates...`);
    
    const response = await new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        console.error('⏰ Duplicate check timeout after 15 seconds');
        resolve({ success: false, error: 'Timeout', duplicates: leads.map(() => false) });
      }, 15000);
      
      try {
        if (!isExtensionContextValid()) {
          clearTimeout(timeout);
          resolve({ success: false, error: 'Extension context invalid', duplicates: leads.map(() => false) });
          return;
        }
      
        chrome.runtime.sendMessage({ 
          action: 'checkDuplicates', 
          leads: leads
        }, (response) => {
          clearTimeout(timeout);
          if (chrome.runtime.lastError) {
            const errorMsg = chrome.runtime.lastError.message || '';
            if (errorMsg.includes('message channel closed') || errorMsg.includes('Extension context invalidated')) {
              console.warn('⚠️ Message channel closed - assuming all leads are NEW');
            } else {
              console.error('❌ Duplicate check runtime error:', chrome.runtime.lastError);
            }
            resolve({ success: false, error: chrome.runtime.lastError.message, duplicates: leads.map(() => false) });
          } else {
            resolve(response || { success: false, duplicates: leads.map(() => false) });
          }
        });
      } catch (error) {
        clearTimeout(timeout);
        const errorMsg = error.message || '';
        if (errorMsg.includes('message channel closed') || errorMsg.includes('Extension context invalidated')) {
          console.warn('⚠️ Message channel closed - assuming all leads are NEW');
        } else {
          console.error('❌ Error calling chrome.runtime.sendMessage:', error);
        }
        resolve({ success: false, error: error.message, duplicates: leads.map(() => false) });
      }
    });
    
    if (response && response.success && Array.isArray(response.duplicates)) {
      if (response.duplicates.length === leads.length) {
        const duplicateCount = response.duplicates.filter(d => d === true).length;
        console.log(`✅ Duplicate check complete: ${duplicateCount} duplicates found out of ${leads.length} leads`);
        return response.duplicates;
      } else {
        console.error(`❌ Duplicate check result length mismatch`);
        return leads.map(() => false);
      }
    } else {
      console.error('❌ Duplicate check failed or returned invalid response');
      return leads.map(() => false);
    }
    
  } catch (error) {
    console.error('❌ Duplicate check exception:', error);
    return leads.map(() => false);
  }
}

async function sendLeadsToServer(leads, workerID) {
  const maxRetries = 3;
  const retryDelay = 2000;
  
  if (!isExtensionContextValid()) {
    if (!sendLeadsToServer._contextWarningLogged) {
      console.warn('⚠️ Extension context invalid - cannot send to server');
      sendLeadsToServer._contextWarningLogged = true;
      setTimeout(() => { sendLeadsToServer._contextWarningLogged = false; }, 5000);
    }
    return 0;
  }
  
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      if (attempt > 1) {
        console.log(`🔄 Retry attempt ${attempt}/${maxRetries}...`);
        await new Promise(resolve => setTimeout(resolve, retryDelay));
      }
      
      if (!isExtensionContextValid()) {
        console.warn('⚠️ Extension context became invalid during retry');
        return 0;
      }
      
      console.log(`📤 Sending ${leads.length} leads to server (Worker: ${workerID})... [Attempt ${attempt}/${maxRetries}]`);
      
      const leadsWithWorkerID = leads.map(lead => ({
        ...lead,
        workerId: workerID || 'Unknown'
      }));
    
      const response = await new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error('Server send timeout'));
        }, 45000);
        
        try {
          if (!isExtensionContextValid()) {
            clearTimeout(timeout);
            reject(new Error('Extension context invalid'));
            return;
          }
      
          chrome.runtime.sendMessage({ 
            action: 'sendToServer', 
            payload: {
              leads: leadsWithWorkerID,
              workerID: workerID
            }
          }, (response) => {
            clearTimeout(timeout);
            if (chrome.runtime.lastError) {
              const errorMsg = chrome.runtime.lastError.message || '';
              if (errorMsg.includes('message channel closed') || errorMsg.includes('Extension context invalidated')) {
                reject(new Error('Message channel closed'));
              } else {
                reject(new Error(chrome.runtime.lastError.message));
              }
            } else {
              resolve(response);
            }
          });
        } catch (error) {
          clearTimeout(timeout);
          reject(error);
        }
      });
      
      if (response && response.success) {
        const insertedCount = response.insertedCount || 0;
        console.log(`✅ Server save: ${insertedCount} leads saved`);
        return insertedCount;
      } else {
        const errorMsg = response?.error || 'Unknown error';
        console.error(`❌ Server save failed: ${errorMsg}`);
        
        if (attempt < maxRetries) {
          continue;
        }
        return 0;
      }
    } catch (error) {
      console.error(`❌ Server send error (attempt ${attempt}/${maxRetries}):`, error.message);
      
      if (attempt < maxRetries) {
        continue;
      }
      
      return 0;
    }
  }
  
  return 0;
}

function getCurrentPageNumber() {
  try {
    const urlParams = new URLSearchParams(window.location.search);
    const pageFromUrl = urlParams.get('page');
    if (pageFromUrl && !isNaN(parseInt(pageFromUrl))) {
      return parseInt(pageFromUrl);
    }
    
    const hashMatch = window.location.href.match(/[#&]page=(\d+)/);
    if (hashMatch) {
      return parseInt(hashMatch[1]);
    }
    
    const currentPageElement = document.querySelector('.artdeco-pagination [aria-current="true"]');
    if (currentPageElement) {
      const pageNum = parseInt(currentPageElement.textContent.trim());
      if (!isNaN(pageNum)) {
        return pageNum;
      }
    }
    
    return 1;
  } catch (error) {
    return 1;
  }
}

async function waitForNewPageLoadComplete(expectedPageNum) {
  console.log(`⏳ Waiting for page ${expectedPageNum} to load...`);
  
  return new Promise((resolve) => {
    let attempts = 0;
    const maxAttempts = 20;
    
    const checkInterval = setInterval(() => {
      attempts++;
      
      const currentPageElement = document.querySelector('[aria-current="true"]');
      const actualPageNum = currentPageElement ? parseInt(currentPageElement.textContent) : 0;
      
      const hasProfiles = document.querySelectorAll('li.artdeco-list__item').length > 0;
      const documentReady = document.readyState === 'complete';
      const hasBasicContent = document.querySelectorAll('li.artdeco-list__item [data-x-search-result="LEAD"]').length > 0;
      
      if (actualPageNum === expectedPageNum && hasProfiles && documentReady && hasBasicContent) {
        clearInterval(checkInterval);
        console.log(`✅ Page ${expectedPageNum} loaded`);
        resolve(true);
        return;
      }
      
      if (attempts >= maxAttempts) {
        clearInterval(checkInterval);
        console.log(`⏰ Page load timeout, proceeding anyway`);
        resolve(true);
        return;
      }
    }, 200);
  });
}

async function goToNextPage(currentPageNum) {
  try {
    const nextPageNum = currentPageNum + 1;
    
    const paginationElement = document.querySelector('.artdeco-pagination');
    if (paginationElement) {
      paginationElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
      await new Promise(resolve => setTimeout(resolve, 1000));
    }
    
    let nextButton = document.querySelector(`button[aria-label="Page ${nextPageNum}"]:not([disabled])`);
    
    if (!nextButton) {
      nextButton = document.querySelector(`[data-test-pagination-page-btn="${nextPageNum}"] button:not([disabled])`);
    }
    
    if (!nextButton) {
      const allButtons = document.querySelectorAll('.artdeco-pagination button:not([disabled])');
      for (const btn of allButtons) {
        if (btn.textContent.trim() === nextPageNum.toString()) {
          nextButton = btn;
          break;
        }
      }
    }
    
    if (!nextButton) {
      nextButton = document.querySelector('button[aria-label="Next"]:not([disabled])');
    }
    
    if (!nextButton || nextButton.disabled) {
      console.log('❌ No next page button found');
      return false;
    }
    
    document.querySelectorAll('[data-processed]').forEach(el => {
      el.removeAttribute('data-processed');
    });
    
    console.log('🔄 Clicking next page button');
    nextButton.click();
    
    const pageLoaded = await waitForNewPageLoadComplete(nextPageNum);
    
    if (pageLoaded) {
      console.log('📍 Applying ULTRA scroll to new page');
      await ensureAtTop();
      
      const verificationStart = Date.now();
      while (Date.now() - verificationStart < 2000) {
        const scrollPos = window.pageYOffset || document.documentElement.scrollTop;
        if (scrollPos > 2) {
          console.log(`⚠️ Drift detected (${scrollPos}px), correcting...`);
          window.scrollTo(0, 0);
          document.documentElement.scrollTop = 0;
        }
        await new Promise(resolve => setTimeout(resolve, 100));
      }
      
      console.log('✅ New page verified and locked at top');
    }
    
    return pageLoaded;
    
  } catch (error) {
    console.error('Error navigating to next page:', error);
    return false;
  }
}

// MAIN EXTRACTION FUNCTION - Server Mode
async function extractLeadsToServer(maxPages, workerID, country) {
  // Check for saved leads from previous run FIRST - AUTO RECOVERY
  const savedData = loadSavedLeadsFromLocalStorage();
  if (savedData && savedData.leads && savedData.leads.length > 0) {
    console.log(`\n${'='.repeat(80)}`);
    console.log(`💾 AUTO-RECOVERY: Found ${savedData.count} saved leads from previous extraction`);
    console.log(`   📊 Pages Processed: ${savedData.pagesProcessed || 'N/A'}`);
    console.log(`   👤 Worker ID: ${savedData.workerID}`);
    console.log(`   🕐 Timestamp: ${savedData.timestamp}`);
    console.log(`${'='.repeat(80)}\n`);
    
    if (isExtensionContextValid()) {
      console.log(`🔄 Attempting to recover and upload saved leads...`);
      
      try {
        const savedLeads = savedData.leads.map((lead, index) => ({
          ...lead,
          leadId: lead.leadId || `LEAD_${Date.now()}_${index}`,
          country: lead.country || savedData.country || 'Unknown',
          workerId: savedData.workerID || workerID,
          extractedAt: lead.extractedAt || new Date().toISOString()
        }));
        
        const leadsForDupCheck = savedLeads.map(lead => ({
          fullName: (lead.fullName || '').trim(),
          company: (lead.company || '').trim(),
          country: (lead.country || savedData.country || 'Unknown').trim(),
          state: (lead.state || '').trim()
        }));
        
        if (!isExtensionContextValid()) {
          console.warn(`⚠️ Extension context invalid during recovery`);
          throw new Error('Extension context invalid during recovery');
        }
        
        console.log(`🔍 Checking ${leadsForDupCheck.length} saved leads for duplicates...`);
        const duplicateResults = await checkPageDuplicates(leadsForDupCheck);
        
        const finalSavedLeads = [];
        let savedDuplicates = 0;
        for (let i = 0; i < savedLeads.length; i++) {
          const lead = savedLeads[i];
          const isDuplicate = duplicateResults[i] === true;
          
          if (!lead.fullName || !lead.company) {
            continue;
          }
          
          if (!isDuplicate) {
            finalSavedLeads.push(lead);
          } else {
            savedDuplicates++;
          }
        }
        
        if (finalSavedLeads.length > 0) {
          console.log(`\n${'='.repeat(80)}`);
          console.log(`💾 RECOVERY: ${finalSavedLeads.length} new leads recovered (${savedDuplicates} duplicates skipped)`);
          console.log(`${'='.repeat(80)}\n`);
          
          if (!isExtensionContextValid()) {
            console.warn(`⚠️ Extension context invalid during recovery - skipping upload`);
            throw new Error('Extension context invalid during recovery upload');
          }
          
          const savedCount = await sendLeadsToServer(finalSavedLeads, savedData.workerID || workerID);
          
          console.log(`\n${'='.repeat(80)}`);
          console.log(`✅ RECOVERY COMPLETE: ${savedCount} leads uploaded from previous extraction`);
          console.log(`${'='.repeat(80)}\n`);
          
          clearSavedLeadsFromLocalStorage();
          console.log(`🗑️ Cleared saved leads from localStorage`);
        } else {
          console.log(`⚠️ All saved leads were duplicates - clearing backup`);
          clearSavedLeadsFromLocalStorage();
        }
      } catch (error) {
        console.error(`❌ Failed to recover saved leads:`, error);
        console.log(`⚠️ Saved leads will be kept for next attempt`);
      }
    } else {
      console.warn(`⚠️ Extension context invalid - cannot recover saved leads now`);
    }
  }
  
  // Validate extension context before starting new extraction
  if (!isExtensionContextValid()) {
    const errorMsg = '❌ Extension context invalid - chrome.runtime is not available. This usually means the extension was reloaded. Please refresh the LinkedIn page and try again.';
    console.error(errorMsg);
    alert(errorMsg);
    throw new Error('Extension context invalid - please refresh the page');
  }
  
  console.log(`🚀 FAST MODE: Starting extraction for ${maxPages} pages - Worker: ${workerID}`);
  
  const allLeads = [];
  let totalDuplicates = 0;
  let currentPage = getCurrentPageNumber();
  let pagesProcessed = 0;
  
  console.log(`📍 STARTING PAGE DETECTION: Currently on page ${currentPage}`);
  console.log(`📍 EXTRACTION PLAN: Will extract pages ${currentPage} to ${currentPage + maxPages - 1}`);
  
  addStatsIndicator(maxPages);
  
  try {
    // PHASE 1: RAPID EXTRACTION
    console.log('🏃‍♂️ PHASE 1: Rapid extraction - collecting all leads first');
    
    for (let i = 0; i < maxPages; i++) {
      if (i > 0 && i % 3 === 0) {
        if (!isExtensionContextValid()) {
          const errorMsg = `❌ Extension context became invalid during extraction (page ${currentPage}). Please refresh the LinkedIn page and try again.`;
          console.error(errorMsg);
          console.error(`⚠️ Collected ${allLeads.length} leads before context was lost`);
          alert(errorMsg + `\n\nCollected ${allLeads.length} leads before error. Please refresh and try again.`);
          throw new Error('Extension context invalid during extraction');
        }
      }
      
      console.log(`⚡ FAST: Processing page ${currentPage} (${i + 1}/${maxPages})`);
      const currentStats = { new: allLeads.length, duplicates: totalDuplicates, failed: 0 };
      updateStats(i + 1, maxPages, `⚡ FAST: Extracting page ${currentPage}...`, 0, currentStats);
      
      const pageResult = await extractPageFast(currentPage);
      
      if (pageResult.newLeads.length === 0 && pageResult.duplicates === 0) {
        console.log(`No leads found on page ${currentPage}, stopping`);
        break;
      }
      
      console.log(`⚡ FAST Page ${currentPage}: ${pageResult.newLeads.length} NEW, ${pageResult.duplicates} duplicates`);
      
      allLeads.push(...pageResult.newLeads);
      pagesProcessed = i + 1;
      totalDuplicates += pageResult.duplicates;
      
      if (i > 0 && i % 5 === 0) {
        saveLeadsToLocalStorage(allLeads, workerID, country, currentPage, pagesProcessed);
      }
      
      const fastStats = { new: allLeads.length, duplicates: totalDuplicates, failed: 0 };
      updateStats(i + 1, maxPages, `⚡ FAST: ${allLeads.length} leads collected`, 0, fastStats);
      
      if (i < maxPages - 1) {
        const hasNextPage = await goToNextPage(currentPage);
        if (!hasNextPage) {
          console.log('No more pages available');
          break;
        }
        currentPage++;
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }
    
    saveLeadsToLocalStorage(allLeads, workerID, country, currentPage, pagesProcessed);
    
    console.log(`⚡ PHASE 1 COMPLETE! ${allLeads.length} leads collected in ${pagesProcessed} pages`);
    
    if (!isExtensionContextValid()) {
      const errorMsg = `❌ Extension context invalid before saving ${allLeads.length} leads. The extension may have been reloaded.`;
      console.error(errorMsg);
      saveLeadsToLocalStorage(allLeads, workerID, country, currentPage, pagesProcessed);
      console.log(`💾 [BACKGROUND] Saving ${allLeads.length} leads to localStorage for recovery`);
      alert(errorMsg + `\n\n✅ ${allLeads.length} leads being saved to localStorage (background).\n\nThey will be automatically recovered when you:\n1. Refresh the LinkedIn page\n2. Reload the extension in chrome://extensions/\n3. Start extraction again`);
      throw new Error('Extension context invalid before Phase 2');
    }
    
    // PHASE 2: BATCH DUPLICATE CHECK & SAVE
    console.log('🔍 PHASE 2: Batch duplicate check and save to server...');
    updateStats(pagesProcessed, maxPages, `🔍 Checking ${allLeads.length} leads for duplicates...`, 0, 
      { new: allLeads.length, duplicates: totalDuplicates, failed: 0 });
    
    let totalSaved = 0;
    let finalNewLeads = [];
    
    if (allLeads.length > 0) {
      const leadsWithCountry = allLeads.map((lead, index) => ({
        ...lead,
        leadId: lead.leadId || `LEAD_${Date.now()}_${index}`,
        country: lead.country || country || 'Unknown',
        workerId: workerID,
        extractedAt: lead.extractedAt || new Date().toISOString()
      }));
      
      const leadsForDupCheck = leadsWithCountry.map(lead => ({
        fullName: (lead.fullName || '').trim(),
        company: (lead.company || '').trim(),
        country: (lead.country || country || 'Unknown').trim(),
        state: (lead.state || '').trim()
      }));
      
      console.log(`🔍 Checking ${leadsForDupCheck.length} leads for duplicates...`);
      
      let duplicateResults = [];
      try {
        duplicateResults = await checkPageDuplicates(leadsForDupCheck);
      } catch (error) {
        console.error(`❌ Duplicate check failed:`, error);
        if (!isExtensionContextValid()) {
          saveLeadsToLocalStorage(allLeads, workerID, country, currentPage, pagesProcessed);
          console.log(`💾 Saved ${allLeads.length} leads to localStorage due to context loss`);
          throw new Error('Extension context lost during duplicate check');
        }
        duplicateResults = leadsForDupCheck.map(() => false);
      }
      
      for (let i = 0; i < leadsWithCountry.length; i++) {
        const lead = leadsWithCountry[i];
        const isDuplicate = duplicateResults[i] === true;
        
        if (!lead.fullName || !lead.company) {
          console.log(`⚠️ Skipping invalid lead: ${lead.fullName || 'N/A'} - ${lead.company || 'N/A'}`);
          continue;
        }
        
        if (!isDuplicate) {
          finalNewLeads.push(lead);
        } else {
          totalDuplicates++;
          console.log(`🔄 BATCH DUPLICATE: ${lead.fullName} - ${lead.company}`);
        }
      }
      
      console.log(`🔍 BATCH CHECK COMPLETE! ${finalNewLeads.length} NEW, ${totalDuplicates} duplicates`);
      updateStats(pagesProcessed, maxPages, `💾 Saving ${finalNewLeads.length} NEW leads to server...`, 0, 
        { new: finalNewLeads.length, duplicates: totalDuplicates, failed: 0 });
      
      if (finalNewLeads.length > 0) {
        try {
          totalSaved = await sendLeadsToServer(finalNewLeads, workerID);
          console.log(`💾 BATCH SAVE COMPLETE! ${totalSaved}/${finalNewLeads.length} leads actually saved to server`);
          
          if (totalSaved > 0) {
            clearSavedLeadsFromLocalStorage();
            console.log(`🗑️ Cleared saved leads from localStorage after successful upload`);
          }
          
          if (totalSaved < finalNewLeads.length) {
            const notSaved = finalNewLeads.length - totalSaved;
            console.warn(`⚠️ WARNING: ${notSaved} leads were not saved`);
          }
        } catch (error) {
          console.error(`❌ Failed to save leads:`, error);
          if (!isExtensionContextValid()) {
            saveLeadsToLocalStorage(finalNewLeads, workerID, country, currentPage, pagesProcessed);
            console.log(`💾 [BACKGROUND] Saving ${finalNewLeads.length} unsaved leads to localStorage`);
            throw error;
          }
        }
      } else {
        console.log(`💾 No new leads to save (all were duplicates)`);
        clearSavedLeadsFromLocalStorage();
      }
    }
    
    console.log(`🎉 EXTRACTION COMPLETE! ${totalSaved} leads saved to server (${totalDuplicates} duplicates skipped)`);
    const completeStats = { new: finalNewLeads.length, duplicates: totalDuplicates, failed: 0 };
    updateStats(pagesProcessed, maxPages, `🎉 COMPLETE! ${totalSaved} leads saved to server`, totalSaved, completeStats);
    
    setTimeout(() => removeStats(), 30000);
    
    return {
      success: true,
      totalLeads: finalNewLeads.length,
      savedCount: totalSaved,
      savedToDatabase: totalSaved,
      duplicatesFound: totalDuplicates,
      pagesProcessed: pagesProcessed
    };
    
  } catch (error) {
    console.error('Extraction error:', error);
    removeStats();
    throw error;
  }
}

console.log('✅ Content script ready - Server mode');
