// LinkedIn Lead Extractor Popup - Server Mode
console.log('🚀 LinkedIn Lead Extractor - Server Mode');

let workerID = null;
let serverUrl = null;
let apiKey = null;

// Load configuration
async function loadConfig() {
  try {
    const result = await chrome.storage.sync.get(['workerID', 'serverUrl', 'apiKey']);
    workerID = result.workerID || null;
    serverUrl = result.serverUrl || null;
    apiKey = result.apiKey || null;
    
    // Update UI
    if (serverUrl) {
      document.getElementById('serverUrlInput').value = serverUrl;
      document.getElementById('serverStatus').textContent = '✅ Connected';
      document.getElementById('serverStatus').className = 'server-status connected';
    } else {
      document.getElementById('serverStatus').textContent = '❌ Not configured';
      document.getElementById('serverStatus').className = 'server-status disconnected';
    }
    
    if (apiKey) {
      document.getElementById('apiKeyInput').value = apiKey;
    }
    
    if (workerID) {
      document.getElementById('workerSetup').style.display = 'none';
      document.getElementById('workerDisplay').style.display = 'block';
      document.getElementById('currentWorkerId').textContent = workerID;
      document.getElementById('extractBtn').disabled = false;
      document.getElementById('downloadBtn').disabled = false;
      await loadStats();
    } else {
      document.getElementById('workerSetup').style.display = 'block';
      document.getElementById('workerDisplay').style.display = 'none';
    }
  } catch (error) {
    console.error('Error loading config:', error);
  }
}

// Save server configuration
async function saveServerConfig() {
  const url = document.getElementById('serverUrlInput').value.trim();
  const key = document.getElementById('apiKeyInput').value.trim();
  
  if (!url) {
    showError('Please enter a server URL');
    return;
  }
  
  try {
    await chrome.storage.sync.set({
      serverUrl: url,
      apiKey: key || null
    });
    
    // Update background script
    await chrome.runtime.sendMessage({
      action: 'setServerConfig',
      serverUrl: url,
      apiKey: key || null
    });
    
    serverUrl = url;
    apiKey = key || null;
    
    document.getElementById('serverStatus').textContent = '✅ Connected';
    document.getElementById('serverStatus').className = 'server-status connected';
    showSuccess('Server configuration saved!');
  } catch (error) {
    showError('Failed to save configuration: ' + error.message);
  }
}

// Set Worker ID
function setWorkerID(newWorkerID) {
  if (!newWorkerID || newWorkerID.trim().length === 0) {
    return false;
  }
  workerID = newWorkerID.trim();
  chrome.storage.sync.set({ workerID: workerID });
  console.log('✅ Worker ID set:', workerID);
  return true;
}

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  await loadConfig();
  setupEventListeners();
  await detectCountryFromLinkedIn();
});

function setupEventListeners() {
  // Server config
  document.getElementById('saveConfigBtn').addEventListener('click', saveServerConfig);
  
  // Worker ID
  document.getElementById('setWorkerIdBtn').addEventListener('click', () => {
    const newWorkerID = document.getElementById('workerIdInput').value.trim();
    if (setWorkerID(newWorkerID)) {
      document.getElementById('workerSetup').style.display = 'none';
      document.getElementById('workerDisplay').style.display = 'block';
      document.getElementById('currentWorkerId').textContent = workerID;
      document.getElementById('extractBtn').disabled = false;
      document.getElementById('downloadBtn').disabled = false;
      loadStats();
      showSuccess('Worker ID set successfully!');
    } else {
      showError('Please enter a valid Worker ID');
    }
  });
  
  // Extract
  document.getElementById('extractBtn').addEventListener('click', extractLinkedInData);
  
  // Download
  document.getElementById('downloadBtn').addEventListener('click', downloadMyData);
}

// Load stats
async function loadStats() {
  if (!workerID || !serverUrl) return;
  
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'getMyStats',
      workerID: workerID
    });
    
    if (response && response.success) {
      document.getElementById('dailyCount').textContent = (response.daily || 0).toLocaleString();
      document.getElementById('weeklyCount').textContent = (response.weekly || 0).toLocaleString();
      document.getElementById('monthlyCount').textContent = (response.monthly || 0).toLocaleString();
    }
  } catch (error) {
    console.error('Failed to load stats:', error);
  }
}

// Extract LinkedIn data
async function extractLinkedInData() {
  if (!workerID) {
    showError('Please set your Worker ID first');
    return;
  }
  
  if (!serverUrl) {
    showError('Please configure server URL first');
    return;
  }
  
  const extractBtn = document.getElementById('extractBtn');
  const pageSelect = document.getElementById('pageSelect');
  const maxPages = parseInt(pageSelect.value) || 1;
  
  extractBtn.disabled = true;
  extractBtn.textContent = '⚡ Starting...';
  showSuccess('Starting extraction... You can close this popup.');
  
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab.url.includes('linkedin.com')) {
      throw new Error('Please navigate to LinkedIn Sales Navigator first');
    }
    
    // Inject content script if needed
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ['content_server.js']
      });
      await new Promise(resolve => setTimeout(resolve, 50));
    } catch (e) {
      // Script might already be loaded
    }
    
    const results = await chrome.tabs.sendMessage(tab.id, {
      action: 'extractLeads',
      maxPages: maxPages,
      workerID: workerID,
      country: 'auto-detect'
    });
    
    if (results && results.success) {
      if (results.savedToDatabase > 0) {
        const duplicatesMsg = results.duplicatesFound > 0 ? ` (${results.duplicatesFound} duplicates skipped)` : '';
        showSuccess(`✅ SUCCESS! ${results.savedToDatabase} leads saved${duplicatesMsg}!`);
        await loadStats();
      } else {
        showSuccess('Extraction completed');
        await loadStats();
      }
    } else {
      throw new Error(results?.error || 'Extraction failed');
    }
  } catch (error) {
    showError(`❌ EXTRACTION FAILED: ${error.message}`);
  } finally {
    extractBtn.disabled = false;
    extractBtn.textContent = '⚡ Start Extraction';
  }
}

// Download data
async function downloadMyData() {
  if (!workerID || !serverUrl) {
    showError('Please configure Worker ID and Server URL first');
    return;
  }
  
  try {
    const downloadBtn = document.getElementById('downloadBtn');
    downloadBtn.disabled = true;
    downloadBtn.textContent = '📥 Downloading...';
    
    const response = await chrome.runtime.sendMessage({
      action: 'downloadMyData',
      workerID: workerID
    });
    
    if (response && response.success && response.data) {
      const csv = convertToCSV(response.data);
      const blob = new Blob([csv], { type: 'text/csv' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `my_leads_${workerID}_${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      showSuccess(`Downloaded ${response.data.length} leads!`);
    } else {
      throw new Error(response?.error || 'No data found');
    }
  } catch (error) {
    showError('Download failed: ' + error.message);
  } finally {
    const downloadBtn = document.getElementById('downloadBtn');
    downloadBtn.disabled = false;
    downloadBtn.textContent = '📥 Download My Data';
  }
}

function convertToCSV(data) {
  if (!data || data.length === 0) return '';
  const headers = ['Lead ID', 'Full Name', 'Company', 'State', 'Country', 'Import Date', 'File ID', 'Worker ID'];
  const rows = [headers.join(',')];
  for (const row of data) {
    const values = headers.map((_, i) => escapeCSV(row[i] || ''));
    rows.push(values.join(','));
  }
  return rows.join('\n');
}

function escapeCSV(value) {
  if (value === null || value === undefined) return '';
  const str = String(value);
  if (str.includes(',') || str.includes('"') || str.includes('\n')) {
    return `"${str.replace(/"/g, '""')}"`;
  }
  return str;
}

async function detectCountryFromLinkedIn() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab || !tab.url || !tab.url.includes('linkedin.com')) return;
    
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ['content_server.js']
      });
      await new Promise(resolve => setTimeout(resolve, 500));
    } catch (e) {
      // Script might already be loaded
    }
  } catch (error) {
    console.error('Country detection failed:', error);
  }
}

function showError(message) {
  const messagesDiv = document.getElementById('messages');
  if (messagesDiv) {
    const div = document.createElement('div');
    div.className = 'error';
    div.textContent = message;
    messagesDiv.appendChild(div);
    setTimeout(() => div.remove(), 10000);
  }
}

function showSuccess(message) {
  const messagesDiv = document.getElementById('messages');
  if (messagesDiv) {
    const div = document.createElement('div');
    div.className = 'success';
    div.textContent = message;
    messagesDiv.appendChild(div);
    setTimeout(() => div.remove(), 8000);
  }
}

console.log('✅ Popup ready - Server mode');

